// NSFE music file emulator

// Game_Music_Emu 0.4.0
#ifndef NSFE_EMU_H
#define NSFE_EMU_H

#include "blargg_common.h"
#include "Nsf_Emu.h"

// Allows reading info from NSFE file without creating emulator
class Nsfe_Info {
public:
	struct header_t
	{
		char tag [4]; // 'N', 'S', 'F', 'E'
	};
	BOOST_STATIC_ASSERT( sizeof (header_t) == 4 );
	
	// Load NSFE info and optionally load file into Nsf_Emu
	blargg_err_t load_file( const char* path, Nsf_Emu* = 0 );
	blargg_err_t load( Data_Reader&, Nsf_Emu* = 0 );
	blargg_err_t load( header_t const&, Data_Reader&, Nsf_Emu* = 0 );
	typedef blargg_err_t (*loader_t)( void* data, Nsf_Emu::header_t const&, Data_Reader& );
	blargg_err_t load( header_t const&, Data_Reader&, void* loader_data, loader_t );
	
	// Information about current file
	struct info_t : Nsf_Emu::header_t
	{
		// These (longer) fields hide those in Nsf_Emu::header_t
		char game      [256];
		char author    [256];
		char copyright [256];
		char dumper    [256];
	};
	const info_t& info() const { return info_; }
	
	// All track indicies are 0-based
	
	// Name of track [i], or "" if none available
	const char* track_name( int i ) const;
	
	// Duration of track [i] in milliseconds, negative if endless, or 0 if none available
	long track_time( int i ) const;
	
	// Optional playlist consisting of track indicies
	int playlist_size() const { return playlist.size(); }
	int playlist_entry( int i ) const { return playlist [i]; }
	
	// Disable use of playlist in NSFE file, remap track numbers using it
	void disable_playlist( bool = true );
	
	// Get track info
	blargg_err_t track_info( track_info_t* out, int track ) const;
	
	// Read track info without loading into emulator
	static blargg_err_t read_info( Data_Reader& in, track_info_t* out, int track = 0 );
	
public:
	Nsfe_Info();
	~Nsfe_Info();
	int track_count() const { return info_.track_count; }
private:
	blargg_vector<char> track_name_data;
	blargg_vector<const char*> track_names;
	blargg_vector<unsigned char> playlist;
	blargg_vector<long> track_times;
	int track_count_;
	info_t info_;
	bool playlist_disabled;
	
	int remap_track( int i ) const;
	friend class Nsfe_Emu;
};

// NSFE player
class Nsfe_Emu : public Nsf_Emu, public Nsfe_Info {
public:
	// See Nsf_Emu.h for further information
	
	Nsfe_Emu( double gain = 1.4 ) : Nsf_Emu( gain ) { }
	
	typedef Nsfe_Info::header_t header_t;
	
	// Load NSFE data
	blargg_err_t load( Data_Reader& r ) { return Nsfe_Info::load( r, this ); }
	
	// Load NSFE using already-loaded header and remaining data
	blargg_err_t load( header_t const& h, Data_Reader& r ) { return Nsfe_Info::load( h, r, this ); }
	
public:
	Nsf_Emu::track_count;
	Nsf_Emu::load_file;
	void disable_playlist( bool = true );
	Music_Emu::track_info;
	Nsfe_Info::read_info;
	blargg_err_t track_info( track_info_t*, int track ) const;
protected:
	void start_track_( int );
};

inline void Nsfe_Emu::disable_playlist( bool b )
{
	Nsfe_Info::disable_playlist( b );
	set_track_count( info().track_count );
}

#endif
